<?php
/**
 * RSKone App - Field Visit Management API
 * Create new field visit
 */

header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

require_once '../../config/database.php';

// Function to validate token
function validateToken($db, $token) {
    $query = "SELECT us.user_id, u.name, u.email, u.role 
              FROM user_sessions us 
              JOIN users u ON us.user_id = u.id 
              WHERE us.token = :token AND us.expires_at > NOW() AND u.status = 'active'";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':token', $token);
    $stmt->execute();
    
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $headers = getallheaders();
    $token = $headers['Authorization'] ?? '';
    
    if (empty($token)) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Authorization token required'
        ]);
        exit;
    }
    
    $data = json_decode(file_get_contents("php://input"), true);
    
    try {
        $database = new Database();
        $db = $database->getConnection();
        
        // Validate token
        $user = validateToken($db, $token);
        if (!$user) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid or expired token'
            ]);
            exit;
        }
        
        // Validate required fields
        $required_fields = ['customer_name', 'customer_phone', 'visit_date', 'visit_time', 'location', 'purpose'];
        foreach ($required_fields as $field) {
            if (empty($data[$field])) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => "Field '$field' is required"
                ]);
                exit;
            }
        }
        
        // Insert new visit
        $query = "INSERT INTO field_visits 
                  (user_id, customer_name, customer_phone, customer_address, visit_date, visit_time, 
                   location, latitude, longitude, purpose, status, notes, created_at) 
                  VALUES 
                  (:user_id, :customer_name, :customer_phone, :customer_address, :visit_date, :visit_time,
                   :location, :latitude, :longitude, :purpose, 'scheduled', :notes, NOW())";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':user_id', $user['user_id']);
        $stmt->bindParam(':customer_name', $data['customer_name']);
        $stmt->bindParam(':customer_phone', $data['customer_phone']);
        $stmt->bindParam(':customer_address', $data['customer_address'] ?? '');
        $stmt->bindParam(':visit_date', $data['visit_date']);
        $stmt->bindParam(':visit_time', $data['visit_time']);
        $stmt->bindParam(':location', $data['location']);
        $stmt->bindParam(':latitude', $data['latitude'] ?? null);
        $stmt->bindParam(':longitude', $data['longitude'] ?? null);
        $stmt->bindParam(':purpose', $data['purpose']);
        $stmt->bindParam(':notes', $data['notes'] ?? '');
        
        if ($stmt->execute()) {
            $visit_id = $db->lastInsertId();
            
            http_response_code(201);
            echo json_encode([
                'success' => true,
                'message' => 'Field visit created successfully',
                'data' => [
                    'visit_id' => $visit_id,
                    'customer_name' => $data['customer_name'],
                    'visit_date' => $data['visit_date'],
                    'visit_time' => $data['visit_time']
                ]
            ]);
        } else {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Failed to create field visit'
            ]);
        }
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Server error: ' . $e->getMessage()
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
}
?>
