<?php
/**
 * RSKone App - Reporting Dashboard API
 * Get dashboard data for field sales team
 */

header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

require_once '../../config/database.php';

// Function to validate token
function validateToken($db, $token) {
    $query = "SELECT us.user_id, u.name, u.email, u.role 
              FROM user_sessions us 
              JOIN users u ON us.user_id = u.id 
              WHERE us.token = :token AND us.expires_at > NOW() AND u.status = 'active'";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':token', $token);
    $stmt->execute();
    
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $headers = getallheaders();
    $token = $headers['Authorization'] ?? '';
    
    if (empty($token)) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Authorization token required'
        ]);
        exit;
    }
    
    try {
        $database = new Database();
        $db = $database->getConnection();
        
        // Validate token
        $user = validateToken($db, $token);
        if (!$user) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid or expired token'
            ]);
            exit;
        }
        
        $current_month = date('Y-m');
        $current_date = date('Y-m-d');
        
        // Get today's visits
        $todayVisitsQuery = "SELECT COUNT(*) as count FROM field_visits 
                            WHERE user_id = :user_id AND visit_date = :current_date";
        $todayStmt = $db->prepare($todayVisitsQuery);
        $todayStmt->bindParam(':user_id', $user['user_id']);
        $todayStmt->bindParam(':current_date', $current_date);
        $todayStmt->execute();
        $today_visits = $todayStmt->fetch(PDO::FETCH_ASSOC)['count'];
        
        // Get monthly statistics
        $monthlyQuery = "SELECT 
                            COUNT(*) as total_visits,
                            COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_visits,
                            COUNT(CASE WHEN status = 'scheduled' THEN 1 END) as scheduled_visits,
                            COUNT(CASE WHEN status = 'cancelled' THEN 1 END) as cancelled_visits
                         FROM field_visits 
                         WHERE user_id = :user_id AND DATE_FORMAT(visit_date, '%Y-%m') = :current_month";
        $monthlyStmt = $db->prepare($monthlyQuery);
        $monthlyStmt->bindParam(':user_id', $user['user_id']);
        $monthlyStmt->bindParam(':current_month', $current_month);
        $monthlyStmt->execute();
        $monthly_stats = $monthlyStmt->fetch(PDO::FETCH_ASSOC);
        
        // Get collection statistics
        $collectionQuery = "SELECT 
                              COUNT(*) as total_collections,
                              COALESCE(SUM(amount), 0) as total_amount,
                              COALESCE(AVG(amount), 0) as avg_amount
                           FROM collections 
                           WHERE user_id = :user_id AND DATE_FORMAT(collection_date, '%Y-%m') = :current_month";
        $collectionStmt = $db->prepare($collectionQuery);
        $collectionStmt->bindParam(':user_id', $user['user_id']);
        $collectionStmt->bindParam(':current_month', $current_month);
        $collectionStmt->execute();
        $collection_stats = $collectionStmt->fetch(PDO::FETCH_ASSOC);
        
        // Get target information
        $targetQuery = "SELECT target_amount, collected_amount, 
                        CASE WHEN collected_amount >= target_amount THEN 'achieved' ELSE 'pending' END as status
                        FROM user_targets 
                        WHERE user_id = :user_id AND target_month = :current_month";
        $targetStmt = $db->prepare($targetQuery);
        $targetStmt->bindParam(':user_id', $user['user_id']);
        $targetStmt->bindParam(':current_month', $current_month);
        $targetStmt->execute();
        $target_info = $targetStmt->fetch(PDO::FETCH_ASSOC);
        
        // Get recent visits
        $recentVisitsQuery = "SELECT id, customer_name, visit_date, visit_time, status, location
                             FROM field_visits 
                             WHERE user_id = :user_id 
                             ORDER BY visit_date DESC, visit_time DESC 
                             LIMIT 5";
        $recentStmt = $db->prepare($recentVisitsQuery);
        $recentStmt->bindParam(':user_id', $user['user_id']);
        $recentStmt->execute();
        $recent_visits = $recentStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get recent collections
        $recentCollectionsQuery = "SELECT id, customer_name, amount, collection_date, payment_method
                                  FROM collections 
                                  WHERE user_id = :user_id 
                                  ORDER BY collection_date DESC 
                                  LIMIT 5";
        $recentCollectionStmt = $db->prepare($recentCollectionsQuery);
        $recentCollectionStmt->bindParam(':user_id', $user['user_id']);
        $recentCollectionStmt->execute();
        $recent_collections = $recentCollectionStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate completion rate
        $completion_rate = $monthly_stats['total_visits'] > 0 
            ? round(($monthly_stats['completed_visits'] / $monthly_stats['total_visits']) * 100, 2)
            : 0;
        
        // Calculate target achievement
        $target_achievement = $target_info && $target_info['target_amount'] > 0
            ? round(($target_info['collected_amount'] / $target_info['target_amount']) * 100, 2)
            : 0;
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'data' => [
                'user' => $user,
                'current_month' => $current_month,
                'today_visits' => $today_visits,
                'monthly_stats' => [
                    'total_visits' => $monthly_stats['total_visits'],
                    'completed_visits' => $monthly_stats['completed_visits'],
                    'scheduled_visits' => $monthly_stats['scheduled_visits'],
                    'cancelled_visits' => $monthly_stats['cancelled_visits'],
                    'completion_rate' => $completion_rate
                ],
                'collection_stats' => [
                    'total_collections' => $collection_stats['total_collections'],
                    'total_amount' => $collection_stats['total_amount'],
                    'avg_amount' => $collection_stats['avg_amount']
                ],
                'target_info' => [
                    'target_amount' => $target_info['target_amount'] ?? 0,
                    'collected_amount' => $target_info['collected_amount'] ?? 0,
                    'achievement_percentage' => $target_achievement,
                    'status' => $target_info['status'] ?? 'no_target'
                ],
                'recent_visits' => $recent_visits,
                'recent_collections' => $recent_collections
            ]
        ]);
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Server error: ' . $e->getMessage()
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
}
?>
