<?php
/**
 * RSKone App - Location Tracking API
 * Track user location for field visits
 */

header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

require_once '../../config/database.php';

// Function to validate token
function validateToken($db, $token) {
    $query = "SELECT us.user_id, u.name, u.email, u.role 
              FROM user_sessions us 
              JOIN users u ON us.user_id = u.id 
              WHERE us.token = :token AND us.expires_at > NOW() AND u.status = 'active'";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':token', $token);
    $stmt->execute();
    
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $headers = getallheaders();
    $token = $headers['Authorization'] ?? '';
    
    if (empty($token)) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Authorization token required'
        ]);
        exit;
    }
    
    $data = json_decode(file_get_contents("php://input"), true);
    
    try {
        $database = new Database();
        $db = $database->getConnection();
        
        // Validate token
        $user = validateToken($db, $token);
        if (!$user) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid or expired token'
            ]);
            exit;
        }
        
        // Validate required fields
        $required_fields = ['latitude', 'longitude'];
        foreach ($required_fields as $field) {
            if (!isset($data[$field]) || !is_numeric($data[$field])) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => "Valid $field is required"
                ]);
                exit;
            }
        }
        
        // Validate latitude and longitude ranges
        if ($data['latitude'] < -90 || $data['latitude'] > 90) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid latitude value'
            ]);
            exit;
        }
        
        if ($data['longitude'] < -180 || $data['longitude'] > 180) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid longitude value'
            ]);
            exit;
        }
        
        // Insert location record
        $query = "INSERT INTO location_tracking (user_id, latitude, longitude, accuracy, timestamp) 
                  VALUES (:user_id, :latitude, :longitude, :accuracy, NOW())";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':user_id', $user['user_id']);
        $stmt->bindParam(':latitude', $data['latitude']);
        $stmt->bindParam(':longitude', $data['longitude']);
        $stmt->bindParam(':accuracy', $data['accuracy'] ?? null);
        
        if ($stmt->execute()) {
            $location_id = $db->lastInsertId();
            
            http_response_code(201);
            echo json_encode([
                'success' => true,
                'message' => 'Location tracked successfully',
                'data' => [
                    'location_id' => $location_id,
                    'latitude' => $data['latitude'],
                    'longitude' => $data['longitude'],
                    'timestamp' => date('Y-m-d H:i:s')
                ]
            ]);
        } else {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Failed to track location'
            ]);
        }
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Server error: ' . $e->getMessage()
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
}
?>
