<?php
/**
 * RSKone App - Incentive Calculation API
 * Calculate user incentives based on performance
 */

header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

require_once '../../config/database.php';

// Function to validate token
function validateToken($db, $token) {
    $query = "SELECT us.user_id, u.name, u.email, u.role 
              FROM user_sessions us 
              JOIN users u ON us.user_id = u.id 
              WHERE us.token = :token AND us.expires_at > NOW() AND u.status = 'active'";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':token', $token);
    $stmt->execute();
    
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

// Function to calculate incentives
function calculateIncentives($db, $user_id, $month = null) {
    if (!$month) {
        $month = date('Y-m');
    }
    
    // Get user's performance data
    $query = "SELECT 
                COUNT(DISTINCT fv.id) as total_visits,
                COUNT(DISTINCT CASE WHEN fv.status = 'completed' THEN fv.id END) as completed_visits,
                COALESCE(SUM(c.amount), 0) as total_collections,
                COALESCE(COUNT(DISTINCT c.id), 0) as collection_count
              FROM field_visits fv
              LEFT JOIN collections c ON c.user_id = fv.user_id 
                AND DATE_FORMAT(c.collection_date, '%Y-%m') = :month
              WHERE fv.user_id = :user_id 
                AND DATE_FORMAT(fv.visit_date, '%Y-%m') = :month";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':user_id', $user_id);
    $stmt->bindParam(':month', $month);
    $stmt->execute();
    
    $performance = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get incentive rates
    $rateQuery = "SELECT * FROM incentive_rates WHERE status = 'active' ORDER BY created_at DESC LIMIT 1";
    $rateStmt = $db->prepare($rateQuery);
    $rateStmt->execute();
    $rates = $rateStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$rates) {
        // Default rates if none configured
        $rates = [
            'visit_rate' => 50,
            'completion_rate' => 100,
            'collection_rate' => 0.02,
            'target_bonus_rate' => 0.05
        ];
    }
    
    // Calculate incentives
    $visit_incentive = $performance['total_visits'] * $rates['visit_rate'];
    $completion_incentive = $performance['completed_visits'] * $rates['completion_rate'];
    $collection_incentive = $performance['total_collections'] * $rates['collection_rate'];
    
    // Check if target achieved
    $targetQuery = "SELECT target_amount, collected_amount FROM user_targets 
                    WHERE user_id = :user_id AND target_month = :month";
    $targetStmt = $db->prepare($targetQuery);
    $targetStmt->bindParam(':user_id', $user_id);
    $targetStmt->bindParam(':month', $month);
    $targetStmt->execute();
    $target = $targetStmt->fetch(PDO::FETCH_ASSOC);
    
    $target_bonus = 0;
    if ($target && $target['collected_amount'] >= $target['target_amount']) {
        $target_bonus = $target['collected_amount'] * $rates['target_bonus_rate'];
    }
    
    $total_incentive = $visit_incentive + $completion_incentive + $collection_incentive + $target_bonus;
    
    return [
        'performance' => $performance,
        'rates' => $rates,
        'incentives' => [
            'visit_incentive' => $visit_incentive,
            'completion_incentive' => $completion_incentive,
            'collection_incentive' => $collection_incentive,
            'target_bonus' => $target_bonus,
            'total_incentive' => $total_incentive
        ],
        'target' => $target
    ];
}

if ($_SERVER['REQUEST_METHOD'] === 'GET' || $_SERVER['REQUEST_METHOD'] === 'POST') {
    $headers = getallheaders();
    $token = $headers['Authorization'] ?? '';
    
    if (empty($token)) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Authorization token required'
        ]);
        exit;
    }
    
    try {
        $database = new Database();
        $db = $database->getConnection();
        
        // Validate token
        $user = validateToken($db, $token);
        if (!$user) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid or expired token'
            ]);
            exit;
        }
        
        // Get month parameter
        $month = $_GET['month'] ?? $_POST['month'] ?? date('Y-m');
        
        // Calculate incentives
        $incentive_data = calculateIncentives($db, $user['user_id'], $month);
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'data' => [
                'user_id' => $user['user_id'],
                'month' => $month,
                'performance' => $incentive_data['performance'],
                'incentives' => $incentive_data['incentives'],
                'target' => $incentive_data['target'],
                'rates' => $incentive_data['rates']
            ]
        ]);
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Server error: ' . $e->getMessage()
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
}
?>
