<?php
/**
 * RSKone App - Collection Management API
 * Create new collection record
 */

header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

require_once '../../config/database.php';

// Function to validate token
function validateToken($db, $token) {
    $query = "SELECT us.user_id, u.name, u.email, u.role 
              FROM user_sessions us 
              JOIN users u ON us.user_id = u.id 
              WHERE us.token = :token AND us.expires_at > NOW() AND u.status = 'active'";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':token', $token);
    $stmt->execute();
    
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $headers = getallheaders();
    $token = $headers['Authorization'] ?? '';
    
    if (empty($token)) {
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Authorization token required'
        ]);
        exit;
    }
    
    $data = json_decode(file_get_contents("php://input"), true);
    
    try {
        $database = new Database();
        $db = $database->getConnection();
        
        // Validate token
        $user = validateToken($db, $token);
        if (!$user) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid or expired token'
            ]);
            exit;
        }
        
        // Validate required fields
        $required_fields = ['customer_name', 'customer_phone', 'amount', 'collection_date', 'payment_method'];
        foreach ($required_fields as $field) {
            if (empty($data[$field])) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => "Field '$field' is required"
                ]);
                exit;
            }
        }
        
        // Validate amount
        if (!is_numeric($data['amount']) || $data['amount'] <= 0) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'Amount must be a positive number'
            ]);
            exit;
        }
        
        // Start transaction
        $db->beginTransaction();
        
        try {
            // Insert collection record
            $query = "INSERT INTO collections 
                      (user_id, customer_name, customer_phone, customer_address, amount, 
                       collection_date, payment_method, reference_number, notes, status, created_at) 
                      VALUES 
                      (:user_id, :customer_name, :customer_phone, :customer_address, :amount,
                       :collection_date, :payment_method, :reference_number, :notes, 'completed', NOW())";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':user_id', $user['user_id']);
            $stmt->bindParam(':customer_name', $data['customer_name']);
            $stmt->bindParam(':customer_phone', $data['customer_phone']);
            $stmt->bindParam(':customer_address', $data['customer_address'] ?? '');
            $stmt->bindParam(':amount', $data['amount']);
            $stmt->bindParam(':collection_date', $data['collection_date']);
            $stmt->bindParam(':payment_method', $data['payment_method']);
            $stmt->bindParam(':reference_number', $data['reference_number'] ?? '');
            $stmt->bindParam(':notes', $data['notes'] ?? '');
            
            $stmt->execute();
            $collection_id = $db->lastInsertId();
            
            // Update user's collection target (if exists)
            $targetQuery = "UPDATE user_targets 
                           SET collected_amount = collected_amount + :amount,
                               updated_at = NOW()
                           WHERE user_id = :user_id AND target_month = DATE_FORMAT(:collection_date, '%Y-%m')";
            
            $targetStmt = $db->prepare($targetQuery);
            $targetStmt->bindParam(':amount', $data['amount']);
            $targetStmt->bindParam(':user_id', $user['user_id']);
            $targetStmt->bindParam(':collection_date', $data['collection_date']);
            $targetStmt->execute();
            
            // Commit transaction
            $db->commit();
            
            http_response_code(201);
            echo json_encode([
                'success' => true,
                'message' => 'Collection recorded successfully',
                'data' => [
                    'collection_id' => $collection_id,
                    'customer_name' => $data['customer_name'],
                    'amount' => $data['amount'],
                    'collection_date' => $data['collection_date']
                ]
            ]);
            
        } catch (Exception $e) {
            $db->rollBack();
            throw $e;
        }
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Server error: ' . $e->getMessage()
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
}
?>
